# %%
"""
Polygon and change line conversions
===================================

This notebook demonstrates the capabilities of WindKit's `polygons_to_lines` and `lines_to_polygons` functions when working with vector maps (e.g., CORINE). It covers data acquisition, conversion between polygon and change-line representations, error handling for common topological issues (holes, overlaps, missing vertices), and performance benchmarking. The resulting maps can be validated using the WAsP Map Editor.

PyWAsP can work with polygon maps directly, but some workflows and analyses in WAsP require change lines. This notebook illustrates how to perform these conversions effectively and analyze the results.
"""

# %%
import geopandas as gpd
import json
import windkit as wk
import pywasp as pw
from pathlib import Path
from windkit.topography.map_conversions import polygons_to_lines, lines_to_polygons

HOMEDIR = Path.home()

# %%
# Acquire the CORINE polygon map
# ------------------------------
#
# We begin by defining a 4x4 km bounding box around the Risø test site (55.694219, 12.088316) using a 2 km buffer. We then retrieve the corresponding vector map.
#
# The correct UTM zone is automatically determined based on the geographic coordinates. WAsP requires metric coordinate systems (like UTM), so we must ensure our data uses one.
#
# When working with vector maps, we use `geopandas` GeoDataFrames to store and manipulate the data. They support all commonly accepted vector geometries (points, lines, polygons) and handle coordinate reference systems (CRS).
#
# We demonstrate converting these polygons into roughness change lines using two configurations:
#
# * **Roughness map type**: The classic WAsP format with `z_0` change lines.
# * **Landcover map type**: Returns landcover ID lines and optionally a landcover table.

bbox = wk.spatial.BBox.utm_bbox_from_geographic_coordinate(12.088316, 55.694219, 2000)
gdf = wk.get_vector_map(bbox)

# %%
# To convert polygons to lines, we use `polygons_to_lines`. The resulting GeoDataFrame `gdf_lines_z0` contains classic WAsP change lines with roughness lengths on the left (`z0_left`) and right (`z0_right`) of each line.

gdf_lines_z0 = polygons_to_lines(gdf, map_type="roughness", return_lctable=False)

# %%
# For a landcover map, specify `map_type="landcover"`. `gdf_lines_lc` will contain change lines with associated landcover IDs.
gdf_lines_lc = polygons_to_lines(gdf, map_type="landcover", return_lctable=False)

# %%
# To retrieve the landcover table, set `return_lctable=True`. The returned `lct` object allows for easy editing or replacement, facilitating custom landcover classifications and sensitivity analyses.
gdf_lines, lct = polygons_to_lines(gdf, map_type="landcover", return_lctable=True)

# %%
# We can rasterize the polygon GeoDataFrame `gdf`. To obtain the associated landcover table, set `return_lctable=True`.
ras = pw.io.vector_to_raster(gdf, 200, map_type="landcover")
ras, lct = pw.io.vector_to_raster(gdf, 200, map_type="landcover", return_lctable=True)
# %%
# Rasterizing both the original polygons and the derived change lines at 200 m resolution should yield consistent results.
ras = pw.io.vector_to_raster(gdf, 200, map_type="roughness")
ras = pw.io.vector_to_raster(gdf_lines_z0, 200, map_type="roughness")

# %%
# Visualizing the outputs: Raster
# -------------------------------
# We can visualize map data in two ways. First, using built-in xarray plotting for the raster representation, providing a quick overview of grid-based roughness values. This can be compared with the vector-based visualization below.

ras.plot()
# %%
# Visualizing the outputs: Vector
# -------------------------------
# Second, we use `windkit.plot.landcover_map` to visualize the original vector polygons, displaying roughness lengths (`z0`) or landcover classes directly from the GeoDataFrame.

wk.plot.landcover_map(gdf, "z0")

# %%
# Reprojection and Benchmarking
# -----------------------------
# We reproject the map to EPSG:32632 to verify coordinate transformation handling. We then perform a "round-trip" test: converting polygons to lines and back to polygons to check data integrity.
#
# Finally, we benchmark the Python implementation against the Fortran-based one, comparing execution times with and without vertex snapping. Performance depends heavily on input geometry size and complexity.

gdf_32632 = gdf.to_crs(32632)
wk.plot.landcover_map(gdf_32632, "z0")
lines, lct = polygons_to_lines(gdf_32632, map_type="landcover", return_lctable=True)
tp_new = lines_to_polygons(lines)

# takes about 31.3 ms
wk.polygons_to_lines(gdf, snap=False, check_errors=False)
# takes about 22 ms
pw.io.polygons_to_lines(gdf, snap=False, check_errors=False)

# takes about 72 ms (most time is spent in snapping)
wk.polygons_to_lines(gdf, snap=True)
# takes about 68 ms (most time is spent in snapping)
pw.io.polygons_to_lines(gdf, snap=True)
# %%
# Handling Partial Metadata: Roughness-only and ID-only
# -----------------------------------------------------
# This section uses a synthetic feature collection to test `polygons_to_lines` with incomplete input data:
#
# * **Roughness only**: Works if `z0` is present, even if IDs are missing.
# * **ID only**: Requires a `LandCoverTable` to map IDs to roughness lengths. We demonstrate passing this table explicitly.
#

area = json.loads("""
{"type": "FeatureCollection", "features": [{"id": "0", "type": "Feature", "properties": {"id": 1, "z0": 0.3, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[311259.8422840722, 6189423.7293760665], [304473.36837506725, 6181958.608076162], [310642.8901105263, 6174740.267645675], [329768.40749044926, 6194482.7371991435], [311259.8422840722, 6189423.7293760665]]]]}}, {"id": "1", "type": "Feature", "properties": {"id": 2, "z0": 0.01, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[337002.1542929823, 6155890.153596819], [297002.1542929823, 6155890.153596819], [297002.1542929823, 6195890.153596819], [337002.1542929823, 6195890.153596819], [337002.1542929823, 6155890.153596819]], [[304473.36837506725, 6181958.608076162], [310642.8901105263, 6174740.267645675], [329768.40749044926, 6194482.7371991435], [311259.8422840722, 6189423.7293760665], [304473.36837506725, 6181958.608076162]]]]}}], "crs": {"type": "name", "properties": {"name": "urn:ogc:def:crs:EPSG::32633"}}}
""")
gdf = gpd.GeoDataFrame.from_features(
    area["features"], crs=area["crs"]["properties"]["name"]
)
wk.plot.landcover_map(gdf, "z0")

gdf_no_id = gdf.drop(columns=["id", "d", "desc"])
lc = polygons_to_lines(gdf_no_id)

gdf_no_z0 = gdf.drop(columns=["z0", "d", "desc"])
# this raises a ValueError because there is no landcover table provided
try:
    lines, lct = polygons_to_lines(gdf_no_z0, map_type="landcover", return_lctable=True)
except KeyError as e:
    print("ValueError:", e)

# this works fine
lines, lct = polygons_to_lines(
    gdf_no_z0,
    lctable=wk.LandCoverTable.get_table("CORINE"),
    map_type="landcover",
    return_lctable=True,
)

# %%
# Topological Edge Case: Polygons with Holes
# ------------------------------------------
# Polygons containing holes can be problematic for WAsP.
#
# *   If `external_roughness` is `None`, conversion raises an error as removing lines enclosing the hole creates an invalid map.
# *   If `external_roughness` is provided, the function warns the user but proceeds, applying the background roughness to the hole.


area = json.loads("""
{"type": "FeatureCollection", "features": [{"id": "0", "type": "Feature", "properties": {"id": 1, "z0": 0.3, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[311259.8422840722, 6189423.7293760665], [304473.36837506725, 6181958.608076162], [310642.8901105263, 6174740.267645675], [329768.40749044926, 6194482.7371991435], [311259.8422840722, 6189423.7293760665]]]]}}, {"id": "1", "type": "Feature", "properties": {"id": 2, "z0": 0.01, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[297002.1542929823, 6195890.153596819], [337002.1542929823, 6195890.153596819], [337002.1542929823, 6155890.153596819], [297002.1542929823, 6155890.153596819], [297002.1542929823, 6195890.153596819]], [[329768.40749044926, 6194482.7371991435], [310025.93793698045, 6190626.786114481], [304473.36837506725, 6181958.608076162], [310642.8901105263, 6174740.267645675], [329768.40749044926, 6194482.7371991435]]]]}}], "crs": {"type": "name", "properties": {"name": "urn:ogc:def:crs:EPSG::32633"}}}
""")
gdf_hole = gpd.GeoDataFrame.from_features(
    area["features"], crs=area["crs"]["properties"]["name"]
)
wk.plot.landcover_map(gdf_hole, "z0")

try:
    lines, lct = polygons_to_lines(
        gdf_hole, external_roughness=None, map_type="landcover", return_lctable=True
    )
except ValueError as e:
    print("ValueError:", e)
# %%
# This works, but with a warning
lines, lct = polygons_to_lines(
    gdf_hole, external_roughness=0.9, map_type="landcover", return_lctable=True
)
lines.plot()

# %%
# Topological Edge Case: Overlapping Polygons
# -------------------------------------------
# Overlapping polygons create ambiguous roughness definitions (line-face inconsistencies). We visualize the overlap using transparency, then perform the conversion. The resulting map is saved to disk and can be opened in the WAsP Map Editor to inspect "Line-Face-Roughness" errors.

area = json.loads("""
{"type": "FeatureCollection", "features": [{"id": "0", "type": "Feature", "properties": {"id": 1, "z0": 0.3, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[311259.8422840722, 6189423.7293760665], [304473.36837506725, 6181958.608076162], [310642.8901105263, 6174740.267645675], [329768.40749044926, 6194482.7371991435], [311259.8422840722, 6189423.7293760665]]]]}}, {"id": "1", "type": "Feature", "properties": {"id": 2, "z0": 0.01, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[297002.1542929823, 6195890.153596819], [337002.1542929823, 6195890.153596819], [337002.1542929823, 6155890.153596819], [297002.1542929823, 6155890.153596819], [297002.1542929823, 6195890.153596819]], [[329768.40749044926, 6194482.7371991435], [314005.27945635153, 6187418.634812043], [304473.36837506725, 6181958.608076162], [310642.8901105263, 6174740.267645675], [329768.40749044926, 6194482.7371991435]]]]}}], "crs": {"type": "name", "properties": {"name": "urn:ogc:def:crs:EPSG::32633"}}}
""")
gdf_overlap = gpd.GeoDataFrame.from_features(
    area["features"], crs=area["crs"]["properties"]["name"]
)
wk.plot.landcover_map(gdf_overlap, "z0", alpha=0.5)

# %%
try:
    lines, lct = polygons_to_lines(
        gdf_overlap, external_roughness=0.9, map_type="landcover", return_lctable=True
    )
except ValueError as e:
    print("ValueError:", e)

wk.landcover_map_to_file(lines, HOMEDIR / "tmp_rou_with_lfr.map", lctable=lct)

# %%
# Topological Edge Case: Missing Vertices (Snapping)
# --------------------------------------------------
# Some geometry sources produce adjacent polygons that do not share vertices at T-junctions.
#
# *   **With Snapping (Default)**: Inserts necessary vertices, creating valid topology.
# *   **Without Snapping**: Runs faster but produces a map with "Cross-point" errors.
#
# We save both outputs to demonstrate the difference.

area = json.loads("""
{"type": "FeatureCollection", "features": [{"id": "0", "type": "Feature", "properties": {"id": 2, "z0": 0.01, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[297002.1542929823, 6195890.153596819], [337002.1542929823, 6195890.153596819], [337002.1542929823, 6155890.153596819], [297002.1542929823, 6155890.153596819], [297002.1542929823, 6195890.153596819]], [[307261.8398003997, 6168139.739435189], [307261.8398003997, 6180999.023657854], [327965.2873988902, 6180999.023657854], [327965.2873988902, 6168139.739435189], [307261.8398003997, 6168139.739435189]]]]}}, {"id": "1", "type": "Feature", "properties": {"id": 3, "z0": 1.1, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[307261.8398003997, 6168139.739435189], [307261.8398003997, 6174569.381546522], [307261.8398003997, 6180999.023657854], [317613.5635996449, 6180999.023657854], [327965.2873988902, 6180999.023657854], [327965.2873988902, 6174569.381546522], [327965.2873988902, 6168139.739435189], [317613.5635996449, 6168139.739435189], [307261.8398003997, 6168139.739435189]]]]}}], "crs": {"type": "name", "properties": {"name": "urn:ogc:def:crs:EPSG::32633"}}}
""")
gdf_missing_vertices = gpd.GeoDataFrame.from_features(
    area["features"], crs=area["crs"]["properties"]["name"]
)
wk.plot.landcover_map(gdf_missing_vertices, "z0")

lines, lct = polygons_to_lines(
    gdf_missing_vertices,
    external_roughness=0.4,
    map_type="landcover",
    return_lctable=True,
)
wk.landcover_map_to_file(lines, HOMEDIR / "tmp_rou_without_crosspoint.map", lctable=lct)

lines, lct = polygons_to_lines(
    gdf_missing_vertices,
    external_roughness=0.4,
    snap=False,
    map_type="landcover",
    return_lctable=True,
)
wk.landcover_map_to_file(lines, HOMEDIR / "tmp_rou_with_crosspoint.map", lctable=lct)

# %%
# Disconnected Polygons and Site Effects
# --------------------------------------
# For maps with disjoint polygons (e.g., islands or specific landcover patches), `external_roughness` must be specified to define the background value. We demonstrate a full workflow: creating a topography map from these polygons and a flat elevation model, then calculating site effects using `pywasp`.

area = json.loads("""
{"type": "FeatureCollection", "features": [{"id": "0", "type": "Feature", "properties": {"id": 1, "z0": 0.05, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[303652.96510379284, 6175776.640083307], [302336.20834703604, 6164743.1265697945], [316139.4515902793, 6158023.1265697945], [329806.47861730633, 6172779.883326551], [303652.96510379284, 6175776.640083307]]]]}}, {"id": "1", "type": "Feature", "properties": {"id": 2, "z0": 0.25, "d": 0.0, "desc": null}, "geometry": {"type": "MultiPolygon", "coordinates": [[[[320362.154292982, 6192712.856299524], [309873.5056443334, 6183858.80224547], [328126.47861730633, 6178864.207650876], [330169.7218605496, 6191713.937380605], [320362.154292982, 6192712.856299524]]]]}}], "crs": {"type": "name", "properties": {"name": "urn:ogc:def:crs:EPSG::32633"}}}
""")
seperate = gpd.GeoDataFrame.from_features(
    area["features"], crs=area["crs"]["properties"]["name"]
)
wk.plot.landcover_map(seperate, "z0")
lines, lct = polygons_to_lines(
    seperate, external_roughness=0.0, map_type="landcover", return_lctable=True
)

# %%
try:
    lines, lct = polygons_to_lines(seperate)
except ValueError as e:
    print("ValueError:", e)
# %%
# Using an external roughness length in `get_site_effects`
# ------------------------------------------------------------
# Note that a warning is issued because the polygons are disconnected. We then create a flat elevation map (`elev`) and use it with the polygon roughness map to compute site effects at specified locations `ds`.

bb = wk.spatial.BBox.from_cornerpts(
    302336.20834704,
    6158023.12656979,
    330169.72186055,
    6192712.85629952,
    crs=seperate.crs,
)
ds = bb.to_grid(20000, 50)
elev = wk.create_vector_map(bbox=bb, map_type="elevation", elevation=0.0)

topo = pw.wasp.TopographyMap(elev, seperate, external_roughness=0.0)
se = topo.get_site_effects(ds, n_sectors=12)

# %%
# Creating a Homogeneous Offshore Map
# -----------------------------------
# For offshore wind resource assessments, it is often necessary to create a "trivial" map with zero elevation and constant roughness. This cell demonstrates creating such a map from scratch using a bounding box, converting it to a raster, and generating the corresponding polygon representation.


bbox = wk.spatial.BBox.utm_bbox_from_geographic_coordinate(7, 55, 20000)
offshore_dtu_elev = wk.create_vector_map(bbox=bbox, elevation=0)
offshore_dtu_rou = wk.create_vector_map(
    bbox=bbox, map_type="roughness", roughness_change=(0.0, 0.0)
)
z0 = wk.lines_to_polygons(offshore_dtu_rou)
wk.plot.landcover_map(z0, "z0")
