"""
TSWC to BWC
=======================

This example demonstrates how to convert a time series wind climate (TSWC) object
to a binned wind climate (BWC) object using the `bwc_from_tswc` function from the
windkit library.

"""

# %%
# Prepare Sample Data
# ------------------------
# First, we need to prepare the sample data. This is done by creating a
# DataFrame with the wind speed and direction data.

import numpy as np
import pandas as pd
import windkit as wk

size = 8760  # 1 year of hourly data

df = pd.DataFrame(
    {
        "wspd_50": np.random.uniform(5, 15, size=size),  # Wind speed in m/s
        "wdir_50": np.random.uniform(0, 360, size=size),  # Wind direction in degrees
    },
    index=pd.date_range("2023-01-01", periods=size, freq="h"),
)
df.index.name = "time"  # Set the index name to 'time'
print(df.head())

# %%
# Create TSWC
# ------------------------

tswc = wk.tswc_from_dataframe(
    df,
    west_east=0.0,
    south_north=0.0,
    crs=4326,
    height_to_columns={50: ["wspd_50", "wdir_50"]},
)
print(tswc)

# %%
# Convert TSWC to BWC
# ------------------------
# Now, we can convert the time series wind climate (TSWC) object to a binned wind climate (BWC) object

bwc = wk.bwc_from_tswc(tswc)
print(bwc)

bwc.wsfreq.sel(sector=270).plot()
